<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Music Supervisor Contact Finder</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            padding: 40px;
        }
        h1 { color: #333; margin-bottom: 10px; font-size: 28px; }
        .subtitle { color: #666; margin-bottom: 30px; font-size: 14px; }
        
        .form-group { margin-bottom: 25px; }
        label {
            display: block;
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
            font-size: 14px;
        }
        .helper-text { font-size: 12px; color: #666; margin-top: 4px; }
        input[type="text"], textarea, select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 6px;
            font-size: 14px;
            transition: border-color 0.3s;
            font-family: inherit;
            background-color: white;
        }
        input[type="text"]:focus, textarea:focus, select:focus {
            outline: none;
            border-color: #667eea;
        }
        select {
            cursor: pointer;
            appearance: none;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' viewBox='0 0 12 12'%3E%3Cpath fill='%23333' d='M6 9L1 4h10z'/%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 12px center;
            padding-right: 36px;
        }
        textarea { resize: vertical; min-height: 80px; }
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        
        .progress-bar {
            width: 100%;
            height: 8px;
            background: #e0e0e0;
            border-radius: 10px;
            overflow: hidden;
            margin: 20px 0;
        }
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #667eea, #764ba2);
            width: 0%;
            transition: width 0.3s;
        }
        .progress-text {
            text-align: center;
            color: #666;
            margin-top: 10px;
            font-size: 14px;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 14px 24px;
            border: none;
            border-radius: 6px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            width: 100%;
            margin: 20px 0;
        }
        .btn-primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
        }
        .btn-primary:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .artists-preview {
            background: #f8f9ff;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
            display: none;
        }
        .artists-preview.show { display: block; }
        .artists-preview h3 {
            color: #667eea;
            margin-bottom: 15px;
            font-size: 18px;
        }
        .artist-chip {
            display: inline-block;
            background: white;
            padding: 8px 15px;
            border-radius: 20px;
            margin: 5px;
            font-size: 14px;
            border: 1px solid #e0e0e0;
        }
        .artist-chip.processing {
            background: #fff3cd;
            border-color: #ffc107;
        }
        .artist-chip.completed {
            background: #d4edda;
            border-color: #28a745;
        }
        
        .supervisors-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }
        .supervisor-card {
            background: #f8f9ff;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            padding: 20px;
        }
        .supervisor-header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 15px;
        }
        .supervisor-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            object-fit: cover;
            background: #667eea;
        }
        .supervisor-name {
            font-size: 18px;
            font-weight: 700;
            color: #333;
        }
        .supervisor-username {
            font-size: 14px;
            color: #666;
        }
        .supervisor-meta {
            font-size: 12px;
            color: #999;
            margin-top: 10px;
        }
        .contact-info {
            background: white;
            padding: 15px;
            border-radius: 6px;
            margin-top: 15px;
        }
        .contact-item {
            display: flex;
            align-items: center;
            gap: 10px;
            margin: 8px 0;
            font-size: 14px;
        }
        .contact-label {
            font-weight: 600;
            color: #667eea;
            min-width: 80px;
        }
        .contact-value {
            color: #333;
            word-break: break-all;
        }
        .social-links {
            display: flex;
            gap: 10px;
            margin-top: 10px;
            flex-wrap: wrap;
        }
        .social-link {
            padding: 6px 12px;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            font-size: 12px;
        }
        .error-message {
            background: #fee;
            border: 1px solid #fcc;
            color: #c33;
            padding: 15px;
            border-radius: 6px;
            margin: 20px 0;
        }
        .success-message {
            background: #efe;
            border: 1px solid #cfc;
            color: #3c3;
            padding: 15px;
            border-radius: 6px;
            margin: 20px 0;
        }
        .stats-bar {
            display: flex;
            justify-content: space-around;
            background: #f8f9ff;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
        }
        .stat-item {
            text-align: center;
        }
        .stat-value {
            font-size: 24px;
            font-weight: 700;
            color: #667eea;
        }
        .stat-label {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🎵 Music Supervisor Contact Finder</h1>
        <p class="subtitle">Find music supervisors and their contact information from artists matching your criteria</p>
        
        <form id="promptForm">
            <div class="form-group">
                <label for="soundDescription">Describe the sound (free text)</label>
                <textarea id="soundDescription" placeholder="Example: dreamy, ethereal vocals with ambient synths..."></textarea>
                <div class="helper-text">Describe the sound in your own words</div>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label for="mood">Mood (optional)</label>
                    <select id="mood" name="mood">
                        <option value="">-- Select a mood --</option>
                        <option value="energetic and uplifting">Energetic & Uplifting</option>
                        <option value="melancholic and introspective">Melancholic & Introspective</option>
                        <option value="aggressive and intense">Aggressive & Intense</option>
                        <option value="calm and relaxing">Calm & Relaxing</option>
                        <option value="dark and moody">Dark & Moody</option>
                        <option value="romantic and sensual">Romantic & Sensual</option>
                        <option value="playful and fun">Playful & Fun</option>
                        <option value="nostalgic and bittersweet">Nostalgic & Bittersweet</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="genre">Genre (optional)</label>
                    <select id="genre" name="genre">
                        <option value="">-- Select a genre --</option>
                        <option value="indie pop">Indie Pop</option>
                        <option value="electronic/synth pop">Electronic/Synth Pop</option>
                        <option value="alternative rock">Alternative Rock</option>
                        <option value="dream pop/shoegaze">Dream Pop/Shoegaze</option>
                        <option value="r&b/soul">R&B/Soul</option>
                        <option value="hip hop/rap">Hip Hop/Rap</option>
                        <option value="folk/acoustic">Folk/Acoustic</option>
                        <option value="jazz/experimental">Jazz/Experimental</option>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label for="additionalInfo">Additional descriptors (optional)</label>
                <input type="text" id="additionalInfo" placeholder="Example: with heavy bass, female vocals, 80s influence..."/>
            </div>
            
            <button type="button" class="btn-primary" id="startBtn">🎨 Find Supervisors from These Artists</button>
        </form>
        
        <div class="artists-preview" id="artistsPreview">
            <h3>Artists Found (<span id="artistCount">0</span>)</h3>
            <div id="artistChips"></div>
        </div>
        
        <div class="stats-bar" style="display: none;" id="statsBar">
            <div class="stat-item">
                <div class="stat-value" id="statArtists">0</div>
                <div class="stat-label">Artists Processed</div>
            </div>
            <div class="stat-item">
                <div class="stat-value" id="statShows">0</div>
                <div class="stat-label">Shows Analyzed</div>
            </div>
            <div class="stat-item">
                <div class="stat-value" id="statSupervisors">0</div>
                <div class="stat-label">Supervisors Found</div>
            </div>
            <div class="stat-item">
                <div class="stat-value" id="statContacts">0</div>
                <div class="stat-label">With Email</div>
            </div>
        </div>
        
        <div class="progress-bar">
            <div class="progress-fill" id="progressFill"></div>
        </div>
        <div class="progress-text" id="progressText">Ready to start</div>
        
        <div id="messages"></div>
        
        <div class="supervisors-grid" id="supervisorsGrid"></div>
    </div>

    <script>
        const startBtn = document.getElementById('startBtn');
        const soundDescription = document.getElementById('soundDescription');
        const mood = document.getElementById('mood');
        const genre = document.getElementById('genre');
        const additionalInfo = document.getElementById('additionalInfo');
        const progressFill = document.getElementById('progressFill');
        const progressText = document.getElementById('progressText');
        const supervisorsGrid = document.getElementById('supervisorsGrid');
        const messagesDiv = document.getElementById('messages');
        const artistsPreview = document.getElementById('artistsPreview');
        const artistChips = document.getElementById('artistChips');
        const artistCount = document.getElementById('artistCount');
        const statsBar = document.getElementById('statsBar');
        const statArtists = document.getElementById('statArtists');
        const statShows = document.getElementById('statShows');
        const statSupervisors = document.getElementById('statSupervisors');
        const statContacts = document.getElementById('statContacts');

        let foundSupervisors = [];
        let processedSupervisorIds = new Set();
        let stats = { artists: 0, shows: 0, supervisors: 0, contacts: 0 };
        const TARGET_SUPERVISORS = 30;
        const CONCURRENT_ARTISTS = 7; // Process 7 artists at once for fetch_artist_shows
        const CONCURRENT_SHOWS = 3;   // Process 3 shows at once for fetch_show_supervisors
        const MAX_RETRIES = 2;

        function formatArtistName(artistName) {
            // Replace spaces with dashes for the API
            return artistName.trim().replace(/\s+/g, '-');
        }

        async function fetchWithRetry(url, options, retries = MAX_RETRIES) {
            for (let attempt = 0; attempt <= retries; attempt++) {
                try {
                    const response = await fetch(url, options);
                    const data = await response.json();
                    
                    if (response.ok && data.success) {
                        return { success: true, data };
                    }
                    
                    // If it's the last attempt, throw error
                    if (attempt === retries) {
                        throw new Error(data.error || 'Request failed');
                    }
                    
                    // Wait a bit before retrying (exponential backoff)
                    await new Promise(resolve => setTimeout(resolve, 1000 * (attempt + 1)));
                    
                } catch (error) {
                    if (attempt === retries) {
                        console.error(`Failed after ${retries + 1} attempts:`, error);
                        return { success: false, error: error.message };
                    }
                    
                    // Wait before retrying
                    await new Promise(resolve => setTimeout(resolve, 1000 * (attempt + 1)));
                }
            }
        }

        function showMessage(message, type = 'info') {
            const div = document.createElement('div');
            div.className = type === 'error' ? 'error-message' : 'success-message';
            div.textContent = message;
            messagesDiv.appendChild(div);
            setTimeout(() => div.remove(), 5000);
        }

        function updateProgress(current, total, message) {
            const percentage = (current / total) * 100;
            progressFill.style.width = percentage + '%';
            progressText.textContent = message;
        }

        function updateStats() {
            statArtists.textContent = stats.artists;
            statShows.textContent = stats.shows;
            statSupervisors.textContent = stats.supervisors;
            statContacts.textContent = stats.contacts;
        }

        function buildPrompt() {
            const freeText = soundDescription.value.trim();
            const moodValue = mood.value;
            const genreValue = genre.value;
            const additional = additionalInfo.value.trim();
            let parts = [];
            if (moodValue) parts.push(moodValue);
            if (genreValue) parts.push(genreValue);
            if (additional) parts.push(additional);
            if (freeText) parts.push(freeText);
            if (parts.length === 0) return '';
            return `Give a list of artists that fit the following description: ${parts.join(' ')}`;
        }

        function addArtistChip(artistName, status = 'pending') {
            const chip = document.createElement('span');
            chip.className = 'artist-chip';
            chip.id = 'artist-' + artistName.replace(/\s/g, '-');
            chip.textContent = artistName;
            if (status === 'processing') chip.classList.add('processing');
            if (status === 'completed') chip.classList.add('completed');
            artistChips.appendChild(chip);
        }

        function updateArtistChip(artistName, status) {
            const chip = document.getElementById('artist-' + artistName.replace(/\s/g, '-'));
            if (chip) {
                chip.classList.remove('processing', 'completed');
                if (status === 'processing') chip.classList.add('processing');
                if (status === 'completed') chip.classList.add('completed');
            }
        }

        function addSupervisorCard(supervisor) {
            const card = document.createElement('div');
            card.className = 'supervisor-card';
            
            let contactHTML = '';
            if (supervisor.contactInfo) {
                contactHTML = `
                    <div class="contact-info">
                        <div class="contact-item">
                            <span class="contact-label">LinkedIn:</span>
                            <a href="${supervisor.contactInfo.linkedin_url}" target="_blank" class="contact-value">${supervisor.contactInfo.linkedin_url}</a>
                        </div>
                        ${supervisor.contactInfo.emails && supervisor.contactInfo.emails.length > 0 ? `
                            <div class="contact-item">
                                <span class="contact-label">Email:</span>
                                <span class="contact-value">${supervisor.contactInfo.emails[0]}</span>
                            </div>
                        ` : ''}
                        ${supervisor.contactInfo.phones && supervisor.contactInfo.phones.length > 0 ? `
                            <div class="contact-item">
                                <span class="contact-label">Phone:</span>
                                <span class="contact-value">${supervisor.contactInfo.phones[0]}</span>
                            </div>
                        ` : ''}
                        ${supervisor.contactInfo.current_employer ? `
                            <div class="contact-item">
                                <span class="contact-label">Company:</span>
                                <span class="contact-value">${supervisor.contactInfo.current_employer}</span>
                            </div>
                        ` : ''}
                        ${supervisor.contactInfo.location ? `
                            <div class="contact-item">
                                <span class="contact-label">Location:</span>
                                <span class="contact-value">${supervisor.contactInfo.location}</span>
                            </div>
                        ` : ''}
                    </div>
                `;
            }

            let socialsHTML = '';
            if (supervisor.profile && supervisor.profile.socials && supervisor.profile.socials.length > 0) {
                socialsHTML = '<div class="social-links">';
                supervisor.profile.socials.forEach(social => {
                    socialsHTML += `<a href="${social.link}" target="_blank" class="social-link">${social.title}</a>`;
                });
                socialsHTML += '</div>';
            }

            card.innerHTML = `
                <div class="supervisor-header">
                    ${supervisor.profile && supervisor.profile.user.customProfileImage ? `
                        <img src="${supervisor.profile.user.customProfileImage}" class="supervisor-avatar" alt="${supervisor.name}">
                    ` : '<div class="supervisor-avatar"></div>'}
                    <div>
                        <div class="supervisor-name">${supervisor.name}</div>
                        <div class="supervisor-username">@${supervisor.profile ? supervisor.profile.user.username : 'N/A'}</div>
                    </div>
                </div>
                <div class="supervisor-meta">
                    ${supervisor.profile ? supervisor.profile.user.userSince : ''} • 
                    From: ${supervisor.foundFrom || 'Unknown Artist'}
                </div>
                ${socialsHTML}
                ${contactHTML}
            `;
            
            supervisorsGrid.appendChild(card);
        }

        async function processSingleShow(show, artist) {
            if (!show.appearanceLink) return null;
            
            stats.shows++;
            updateStats();
            
            try {
                // Fetch show supervisors with retry
                const supervisorsResult = await fetchWithRetry('fetch_show_supervisors.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({ appearance_link: show.appearanceLink })
                });
                
                if (!supervisorsResult.success || !supervisorsResult.data.supervisors) {
                    console.log(`No supervisors found for show ${show.title}, skipping...`);
                    return null;
                }
                
                return {
                    supervisors: supervisorsResult.data.supervisors,
                    artist: artist
                };
                
            } catch (err) {
                console.error('Error fetching show supervisors:', err);
                return null;
            }
        }

        async function processSupervisor(supervisor, artist) {
            if (processedSupervisorIds.has(supervisor.id)) return null;
            
            processedSupervisorIds.add(supervisor.id);
            stats.supervisors++;
            updateStats();
            
            try {
                // Fetch supervisor profile
                const profileResponse = await fetch('fetch_supervisor_profile.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({ slug: supervisor.slug })
                });
                
                const profileData = await profileResponse.json();
                
                // Find LinkedIn and contact info
                const linkedinResponse = await fetch('find_supervisor_linkedin.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({ supervisor_name: supervisor.name })
                });
                
                const linkedinData = await linkedinResponse.json();
                
                // Only return if we got email
                if (linkedinData.success && 
                    linkedinData.contact_info && 
                    linkedinData.contact_info.emails && 
                    linkedinData.contact_info.emails.length > 0) {
                    
                    return {
                        ...supervisor,
                        profile: profileData.success ? profileData.profile : null,
                        contactInfo: linkedinData.contact_info,
                        foundFrom: artist.name
                    };
                }
                
                return null;
                
            } catch (err) {
                console.error('Error processing supervisor:', err);
                return null;
            }
        }

        async function processSingleArtist(artist) {
            updateArtistChip(artist.name, 'processing');
            
            try {
                // Format artist name with dashes
                const formattedArtistName = formatArtistName(artist.name);
                
                // Fetch artist shows with retry
                const showsResult = await fetchWithRetry('fetch_artist_shows.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({ artist_name: formattedArtistName })
                });
                
                if (!showsResult.success || !showsResult.data.data.songs) {
                    console.log(`No shows found for ${artist.name}, skipping...`);
                    updateArtistChip(artist.name, 'completed');
                    return [];
                }
                
                const showsData = showsResult.data;
                
                // Collect all shows from all songs
                const allShows = [];
                for (const song of showsData.data.songs) {
                    if (song.shows && song.shows.length > 0) {
                        allShows.push(...song.shows);
                    }
                }
                
                console.log(`Found ${allShows.length} shows for ${artist.name}`);
                
                // Process shows in batches of 3 concurrently
                const foundSupervisorsFromArtist = [];
                
                for (let i = 0; i < allShows.length; i += CONCURRENT_SHOWS) {
                    if (foundSupervisors.length >= TARGET_SUPERVISORS) break;
                    
                    const showBatch = allShows.slice(i, i + CONCURRENT_SHOWS);
                    const showResults = await Promise.all(
                        showBatch.map(show => processSingleShow(show, artist))
                    );
                    
                    // Process supervisors from this batch of shows
                    for (const showResult of showResults) {
                        if (!showResult || !showResult.supervisors) continue;
                        if (foundSupervisors.length >= TARGET_SUPERVISORS) break;
                        
                        // Process supervisors sequentially to avoid overwhelming APIs
                        for (const supervisor of showResult.supervisors) {
                            if (foundSupervisors.length >= TARGET_SUPERVISORS) break;
                            
                            const supervisorInfo = await processSupervisor(supervisor, showResult.artist);
                            
                            if (supervisorInfo) {
                                foundSupervisors.push(supervisorInfo);
                                foundSupervisorsFromArtist.push(supervisorInfo);
                                stats.contacts++;
                                updateStats();
                                addSupervisorCard(supervisorInfo);
                                
                                updateProgress(
                                    foundSupervisors.length, 
                                    TARGET_SUPERVISORS, 
                                    `Found ${foundSupervisors.length}/${TARGET_SUPERVISORS} supervisors with email`
                                );
                            }
                        }
                    }
                }
                
                stats.artists++;
                updateStats();
                updateArtistChip(artist.name, 'completed');
                
                return foundSupervisorsFromArtist;
                
            } catch (err) {
                console.error(`Error processing artist ${artist.name}:`, err);
                updateArtistChip(artist.name, 'completed');
                return [];
            }
        }

        async function processArtistsConcurrently(artists) {
            // Process 7 artists at a time concurrently
            for (let i = 0; i < artists.length; i += CONCURRENT_ARTISTS) {
                if (foundSupervisors.length >= TARGET_SUPERVISORS) break;
                
                const batch = artists.slice(i, i + CONCURRENT_ARTISTS);
                console.log(`Processing batch of ${batch.length} artists concurrently...`);
                
                // Process all 7 artists at once
                await Promise.all(batch.map(artist => processSingleArtist(artist)));
                
                console.log(`Completed batch. Total supervisors found: ${foundSupervisors.length}`);
                
                if (foundSupervisors.length >= TARGET_SUPERVISORS) {
                    break;
                }
            }
        }

        async function findSupervisorsWorkflow() {
            try {
                const prompt = buildPrompt();
                if (!prompt) {
                    showMessage('Please provide at least one search criteria', 'error');
                    return;
                }
                
                startBtn.disabled = true;
                statsBar.style.display = 'flex';
                foundSupervisors = [];
                processedSupervisorIds = new Set();
                stats = { artists: 0, shows: 0, supervisors: 0, contacts: 0 };
                supervisorsGrid.innerHTML = '';
                artistChips.innerHTML = '';
                messagesDiv.innerHTML = '';
                updateStats();
                
                updateProgress(0, TARGET_SUPERVISORS, 'Generating artist list...');
                
                const artistsResponse = await fetch('openai_handler.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({ prompt })
                });
                
                const artistsData = await artistsResponse.json();
                const artists = artistsData.all_artists || [];
                
                if (artists.length === 0) {
                    throw new Error('No artists generated');
                }
                
                artistsPreview.classList.add('show');
                artistCount.textContent = artists.length;
                
                artists.forEach(artist => addArtistChip(artist.name, 'pending'));
                
                updateProgress(0, TARGET_SUPERVISORS, `Processing ${artists.length} artists (7 concurrent, shows 3 by 3)...`);
                
                await processArtistsConcurrently(artists);
                
                startBtn.disabled = false;
                updateProgress(100, 100, `Complete! Found ${foundSupervisors.length} supervisors with email`);
                showMessage(`Successfully found ${foundSupervisors.length} supervisors with contact information!`, 'success');
                
            } catch (error) {
                startBtn.disabled = false;
                showMessage(`Error: ${error.message}`, 'error');
                console.error('Workflow error:', error);
            }
        }

        startBtn.addEventListener('click', findSupervisorsWorkflow);
    </script>
</body>
</html>