<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Artist Prompt Builder</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            padding: 40px;
        }

        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
        }

        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }

        .form-group {
            margin-bottom: 25px;
        }

        label {
            display: block;
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
            font-size: 14px;
        }

        .helper-text {
            font-size: 12px;
            color: #666;
            margin-top: 4px;
        }

        input[type="text"],
        textarea,
        select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 6px;
            font-size: 14px;
            transition: border-color 0.3s;
            font-family: inherit;
            background-color: white;
        }

        input[type="text"]:focus,
        textarea:focus,
        select:focus {
            outline: none;
            border-color: #667eea;
        }

        select {
            cursor: pointer;
            appearance: none;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' viewBox='0 0 12 12'%3E%3Cpath fill='%23333' d='M6 9L1 4h10z'/%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 12px center;
            padding-right: 36px;
        }

        select:hover {
            border-color: #667eea;
        }

        textarea {
            resize: vertical;
            min-height: 80px;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        @media (max-width: 600px) {
            .form-row {
                grid-template-columns: 1fr;
            }
        }

        .prompt-preview {
            background: #f8f9ff;
            border: 2px solid #667eea;
            border-radius: 6px;
            padding: 15px;
            margin-bottom: 25px;
            display: none;
        }

        .prompt-preview.show {
            display: block;
        }

        .prompt-preview-label {
            font-size: 12px;
            color: #667eea;
            font-weight: 600;
            margin-bottom: 8px;
        }

        .prompt-preview-text {
            color: #333;
            font-size: 14px;
            line-height: 1.5;
        }

        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 30px;
        }

        button {
            flex: 1;
            padding: 14px 24px;
            border: none;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            font-family: inherit;
        }

        button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .btn-primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
        }

        .btn-secondary {
            background: #f0f0f0;
            color: #333;
        }

        .btn-secondary:hover:not(:disabled) {
            background: #e0e0e0;
        }

        .loading {
            display: none;
            text-align: center;
            padding: 20px;
            color: #667eea;
        }

        .loading.show {
            display: block;
        }

        .spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #667eea;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto 10px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .error-message,
        .success-message {
            padding: 12px;
            border-radius: 6px;
            margin-bottom: 20px;
            display: none;
        }

        .error-message.show,
        .success-message.show {
            display: block;
        }

        .error-message {
            background-color: #fee;
            border: 1px solid #fcc;
            color: #c33;
        }

        .success-message {
            background-color: #efe;
            border: 1px solid #cfc;
            color: #3c3;
        }

        .results {
            margin-top: 30px;
            display: none;
        }

        .results.show {
            display: block;
        }

        .results-header {
            background: #f8f9ff;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 15px;
        }

        .results-count {
            font-size: 18px;
            font-weight: 600;
            color: #667eea;
        }

        .results-breakdown {
            font-size: 13px;
            color: #666;
            margin-top: 8px;
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }

        .artist-list {
            max-height: 500px;
            overflow-y: auto;
            border: 2px solid #e0e0e0;
            border-radius: 6px;
            padding: 15px;
        }

        .artist-section {
            margin-bottom: 25px;
        }

        .artist-section:last-child {
            margin-bottom: 0;
        }

        .section-header {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 12px;
            padding-bottom: 8px;
            border-bottom: 2px solid #e0e0e0;
        }

        .section-title {
            font-size: 14px;
            font-weight: 700;
            color: #333;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .section-badge {
            display: inline-block;
            padding: 3px 10px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
        }

        .badge-openai {
            background-color: #e8f4f8;
            color: #0066cc;
            border: 1px solid #b3d9ed;
        }

        .badge-lastfm {
            background-color: #ffe8e8;
            color: #cc0000;
            border: 1px solid #ffb3b3;
        }

        .artist-item {
            padding: 10px;
            border-bottom: 1px solid #f0f0f0;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .artist-item:last-child {
            border-bottom: none;
        }

        .artist-source-icon {
            flex-shrink: 0;
            width: 28px;
            height: 28px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 11px;
            font-weight: 700;
        }

        .icon-openai {
            background-color: #0066cc;
            color: white;
        }

        .icon-lastfm {
            background-color: #cc0000;
            color: white;
        }

        .artist-content {
            flex: 1;
        }

        .artist-name {
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }

        .artist-id {
            font-size: 11px;
            color: #888;
            margin-top: 3px;
        }

        .artist-meta {
            font-size: 11px;
            color: #999;
            margin-top: 2px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🎵 Artist Prompt Builder</h1>
        <p class="subtitle">Build a prompt to find artists that match your criteria</p>

        <form id="promptForm">
            <!-- Free Text Description -->
            <div class="form-group">
                <label for="soundDescription">Describe the sound (free text)</label>
                <textarea 
                    id="soundDescription" 
                    placeholder="Example: dreamy, ethereal vocals with ambient synths..."
                ></textarea>
                <div class="helper-text">Describe the sound in your own words</div>
            </div>

            <!-- Mood and Genre Dropdowns in a Row -->
            <div class="form-row">
                <div class="form-group">
                    <label for="mood">Mood (optional)</label>
                    <select id="mood" name="mood">
                        <option value="">-- Select a mood --</option>
                        <option value="energetic and uplifting">Energetic & Uplifting</option>
                        <option value="melancholic and introspective">Melancholic & Introspective</option>
                        <option value="aggressive and intense">Aggressive & Intense</option>
                        <option value="calm and relaxing">Calm & Relaxing</option>
                        <option value="dark and moody">Dark & Moody</option>
                        <option value="romantic and sensual">Romantic & Sensual</option>
                        <option value="playful and fun">Playful & Fun</option>
                        <option value="nostalgic and bittersweet">Nostalgic & Bittersweet</option>
                    </select>
                    <div class="helper-text">Pick a mood to refine your search</div>
                </div>

                <div class="form-group">
                    <label for="genre">Genre (optional)</label>
                    <select id="genre" name="genre">
                        <option value="">-- Select a genre --</option>
                        <option value="indie pop">Indie Pop</option>
                        <option value="electronic/synth pop">Electronic/Synth Pop</option>
                        <option value="alternative rock">Alternative Rock</option>
                        <option value="dream pop/shoegaze">Dream Pop/Shoegaze</option>
                        <option value="r&b/soul">R&B/Soul</option>
                        <option value="hip hop/rap">Hip Hop/Rap</option>
                        <option value="folk/acoustic">Folk/Acoustic</option>
                        <option value="jazz/experimental">Jazz/Experimental</option>
                    </select>
                    <div class="helper-text">Pick a genre to narrow results</div>
                </div>
            </div>

            <!-- Additional Info -->
            <div class="form-group">
                <label for="additionalInfo">Additional descriptors (optional)</label>
                <input 
                    type="text" 
                    id="additionalInfo" 
                    placeholder="Example: with heavy bass, female vocals, 80s influence..."
                />
                <div class="helper-text">Add any other details that might help</div>
            </div>

            <!-- Prompt Preview -->
            <div class="prompt-preview" id="promptPreview">
                <div class="prompt-preview-label">PROMPT PREVIEW</div>
                <div class="prompt-preview-text" id="promptPreviewText"></div>
            </div>

            <!-- Buttons -->
            <div class="button-group">
                <button type="button" class="btn-primary" id="generateBtn">
                    🎨 Generate Artists
                </button>
                <button type="button" class="btn-secondary" id="expandBtn" disabled>
                    🔍 Expand with Related Artists
                </button>
            </div>
        </form>

        <!-- Loading State -->
        <div class="loading" id="loading">
            <div class="spinner"></div>
            <div>Processing your request...</div>
        </div>

        <!-- Error Message -->
        <div class="error-message" id="errorMessage"></div>

        <!-- Success Message -->
        <div class="success-message" id="successMessage"></div>

        <!-- Results -->
        <div class="results" id="results">
            <div class="results-header">
                <div class="results-count" id="resultsCount"></div>
                <div class="results-breakdown" id="resultsBreakdown"></div>
            </div>
            <div class="artist-list" id="artistList"></div>
        </div>
    </div>

    <script>
        // State management
        let currentArtists = [];
        let isExpanded = false;

        // DOM elements
        const form = document.getElementById('promptForm');
        const soundDescription = document.getElementById('soundDescription');
        const mood = document.getElementById('mood');
        const genre = document.getElementById('genre');
        const additionalInfo = document.getElementById('additionalInfo');
        const promptPreview = document.getElementById('promptPreview');
        const promptPreviewText = document.getElementById('promptPreviewText');
        const generateBtn = document.getElementById('generateBtn');
        const expandBtn = document.getElementById('expandBtn');
        const loading = document.getElementById('loading');
        const errorMessage = document.getElementById('errorMessage');
        const successMessage = document.getElementById('successMessage');
        const results = document.getElementById('results');
        const resultsCount = document.getElementById('resultsCount');
        const resultsBreakdown = document.getElementById('resultsBreakdown');
        const artistList = document.getElementById('artistList');

        // Update prompt preview on any input change
        [soundDescription, mood, genre, additionalInfo].forEach(input => {
            input.addEventListener('input', updatePromptPreview);
            input.addEventListener('change', updatePromptPreview);
        });

        function updatePromptPreview() {
            const prompt = buildPrompt();
            if (prompt) {
                promptPreviewText.textContent = prompt;
                promptPreview.classList.add('show');
            } else {
                promptPreview.classList.remove('show');
            }
        }

        function buildPrompt() {
            const freeText = soundDescription.value.trim();
            const moodValue = mood.value;
            const genreValue = genre.value;
            const additional = additionalInfo.value.trim();

            // Build prompt parts
            let parts = [];
            
            if (moodValue) parts.push(moodValue);
            if (genreValue) parts.push(genreValue);
            if (additional) parts.push(additional);
            if (freeText) parts.push(freeText);

            if (parts.length === 0) return '';

            return `Give a list of artists that fit the following description: ${parts.join(' ')}`;
        }

        function showError(message) {
            errorMessage.textContent = message;
            errorMessage.classList.add('show');
            successMessage.classList.remove('show');
        }

        function showSuccess(message) {
            successMessage.textContent = message;
            successMessage.classList.add('show');
            errorMessage.classList.remove('show');
        }

        function hideMessages() {
            errorMessage.classList.remove('show');
            successMessage.classList.remove('show');
        }

        function setLoading(isLoading) {
            if (isLoading) {
                loading.classList.add('show');
                generateBtn.disabled = true;
                expandBtn.disabled = true;
            } else {
                loading.classList.remove('show');
                generateBtn.disabled = false;
                if (currentArtists.length > 0) {
                    expandBtn.disabled = false;
                }
            }
        }

        function displayResults(data) {
            const openaiArtists = data.openai_artists || [];
            const lastfmArtists = data.lastfm_artists || [];
            const allArtists = data.all_artists || [];

            if (allArtists.length === 0) {
                showError('No artists found for this description. Try adjusting mood, genre, or add a bit more detail.');
                results.classList.remove('show');
                return;
            }

            currentArtists = allArtists;
            
            // Update results count
            resultsCount.textContent = `${allArtists.length} artist${allArtists.length !== 1 ? 's' : ''} found`;
            
            // Update breakdown
            resultsBreakdown.innerHTML = `
                <div class="count-item">
                    <span class="section-badge badge-openai">OpenAI: ${openaiArtists.length}</span>
                </div>
                <div class="count-item">
                    <span class="section-badge badge-lastfm">Last.fm: ${lastfmArtists.length}</span>
                </div>
            `;

            // Build artist list with sections
            let html = '';

            // OpenAI section
            if (openaiArtists.length > 0) {
                html += `
                    <div class="artist-section">
                        <div class="section-header">
                            <span class="section-title">AI Recommendations</span>
                            <span class="section-badge badge-openai">${openaiArtists.length}</span>
                        </div>
                        ${openaiArtists.map(artist => createArtistHTML(artist)).join('')}
                    </div>
                `;
            }

            // Last.fm section
            if (lastfmArtists.length > 0) {
                html += `
                    <div class="artist-section">
                        <div class="section-header">
                            <span class="section-title">Similar Artists (Last.fm)</span>
                            <span class="section-badge badge-lastfm">${lastfmArtists.length}</span>
                        </div>
                        ${lastfmArtists.map(artist => createArtistHTML(artist)).join('')}
                    </div>
                `;
            }

            artistList.innerHTML = html;
            results.classList.add('show');
            expandBtn.disabled = false;
        }

        function createArtistHTML(artist) {
            const source = artist.source || 'unknown';
            const iconClass = source === 'openai' ? 'icon-openai' : 'icon-lastfm';
            const iconText = source === 'openai' ? 'AI' : 'FM';
            
            let metaInfo = '';
            if (artist.match) {
                metaInfo += `<div class="artist-meta">Match: ${(artist.match * 100).toFixed(0)}%</div>`;
            }
            
            return `
                <div class="artist-item">
                    <div class="artist-source-icon ${iconClass}">${iconText}</div>
                    <div class="artist-content">
                        <div class="artist-name">${escapeHtml(artist.name)}</div>
                        ${artist.artist_id ? `<div class="artist-id">ID: ${escapeHtml(artist.artist_id)}</div>` : ''}
                        ${metaInfo}
                    </div>
                </div>
            `;
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // Generate Artists button handler
        generateBtn.addEventListener('click', async function() {
            hideMessages();
            
            const prompt = buildPrompt();
            if (!prompt) {
                showError('Please provide at least one input: free text, mood, or genre.');
                return;
            }

            setLoading(true);
            isExpanded = false;

            try {
                const response = await fetch('openai_handler.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ prompt })
                });

                const data = await response.json();

                if (!response.ok) {
                    throw new Error(data.error || 'Failed to generate artists');
                }

                if (data.all_artists && data.all_artists.length > 0) {
                    displayResults(data);
                    const counts = data.counts || {};
                    showSuccess(`Successfully generated ${counts.total || data.all_artists.length} artists! (${counts.openai || 0} from AI, ${counts.lastfm || 0} from Last.fm)`);
                } else {
                    showError('No artists found for this description. Try adjusting mood, genre, or add a bit more detail.');
                }

            } catch (error) {
                showError(`Error: ${error.message}. Please try again.`);
                console.error('Error:', error);
            } finally {
                setLoading(false);
            }
        });

        // Expand with Related Artists button handler
        expandBtn.addEventListener('click', async function() {
            if (currentArtists.length === 0 || isExpanded) {
                return;
            }

            hideMessages();
            setLoading(true);

            try {
                const response = await fetch('expand_artists.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ artists: currentArtists })
                });

                const data = await response.json();

                if (!response.ok) {
                    throw new Error(data.error || 'Failed to expand artists');
                }

                if (data.artists && data.artists.length > 0) {
                    // For expand, we might get the old format, so format it
                    const formattedData = {
                        openai_artists: [],
                        lastfm_artists: data.artists.map(a => ({...a, source: 'lastfm'})),
                        all_artists: data.artists.map(a => ({...a, source: 'lastfm'}))
                    };
                    displayResults(formattedData);
                    showSuccess(`Successfully expanded to ${data.artists.length} artists! ${data.deduped ? '(Deduplicated)' : ''}`);
                    isExpanded = true;
                } else {
                    showError('No related artists found.');
                }

            } catch (error) {
                showError(`Error: ${error.message}. Please try again.`);
                console.error('Error:', error);
            } finally {
                setLoading(false);
            }
        });

        // Initialize prompt preview
        updatePromptPreview();
    </script>
</body>
</html>