<?php
require_once('db.php');

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!isset($data['supervisor_name']) || empty(trim($data['supervisor_name']))) {
    http_response_code(400);
    echo json_encode(['error' => 'supervisor_name is required']);
    exit();
}

$supervisorName = trim($data['supervisor_name']);
$normalizedName = normalizeSupervisorName($supervisorName);
$conn = getDbConnection();

// Check cache first
if ($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT linkedin_url, contact_data, google_results_count, ai_selected_index, has_email, created_at 
            FROM cache_linkedin_contacts 
            WHERE normalized_name = ? 
            AND expires_at > NOW()
            AND success = 1
            LIMIT 1
        ");
        $stmt->execute([$normalizedName]);
        $cached = $stmt->fetch();
        
        if ($cached) {
            updateCacheStats('find_supervisor_linkedin', true);
            $contactInfo = json_decode($cached['contact_data'], true);
            $response = [
                'success' => true,
                'supervisor_name' => $supervisorName,
                'linkedin_found' => !empty($cached['linkedin_url']),
                'contact_info' => $contactInfo,
                'google_results_count' => $cached['google_results_count'],
                'ai_selected_index' => $cached['ai_selected_index'],
                'cached' => true,
                'cached_at' => $cached['created_at']
            ];
            http_response_code(200);
            echo json_encode($response);
            exit();
        }
        
        updateCacheStats('find_supervisor_linkedin', false);
    } catch (PDOException $e) {
        error_log("Cache check failed: " . $e->getMessage());
    }
}

// Cache miss - perform full lookup
$GOOGLE_API_KEY = 'AIzaSyBLxF7lkZ2rZo3jL6nU5xqWKdEQWr3jE7M';
$GOOGLE_CSE_ID = '94b24be1c90c04c10';
$OPENAI_API_KEY = getenv('OPENAI_API_KEY') ?: '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';
$ROCKETREACH_API_KEY = 'c0fc01cbc83f4c60a7df1bc28ed44af7f66b0e79';

// Step 1: Search Google
$searchQuery = "site:linkedin.com/in " . $supervisorName . " music supervisor";
$googleUrl = "https://www.googleapis.com/customsearch/v1?key=" . $GOOGLE_API_KEY 
             . "&cx=" . $GOOGLE_CSE_ID 
             . "&q=" . urlencode($searchQuery) 
             . "&num=10";

$ch = curl_init($googleUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30
]);

$googleResponse = curl_exec($ch);
$googleHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($googleHttpCode !== 200 || !$googleResponse) {
    http_response_code(500);
    echo json_encode(['error' => 'Failed to search Google', 'code' => $googleHttpCode]);
    exit();
}

$googleResults = json_decode($googleResponse, true);

if (!isset($googleResults['items']) || empty($googleResults['items'])) {
    http_response_code(404);
    echo json_encode(['error' => 'No LinkedIn profiles found', 'supervisor_name' => $supervisorName]);
    exit();
}

// Step 2: Use OpenAI to determine best match
$linkedInProfiles = array_map(function($item) {
    return [
        'title' => $item['title'] ?? '',
        'link' => $item['link'] ?? '',
        'snippet' => $item['snippet'] ?? ''
    ];
}, array_slice($googleResults['items'], 0, 5));

$aiPrompt = "I'm searching for a music supervisor named '{$supervisorName}'. Here are LinkedIn profiles from Google search results:\n\n";
foreach ($linkedInProfiles as $idx => $profile) {
    $aiPrompt .= ($idx + 1) . ". Title: {$profile['title']}\n   URL: {$profile['link']}\n   Snippet: {$profile['snippet']}\n\n";
}
$aiPrompt .= "Which profile is most likely the correct person? Respond with ONLY the number (1-5) of the best match, or '0' if none seem correct.";

$aiMessages = [
    ['role' => 'system', 'content' => 'You are an expert at matching LinkedIn profiles to people. Respond with only a number.'],
    ['role' => 'user', 'content' => $aiPrompt]
];

$aiUrl = 'https://api.openai.com/v1/chat/completions';
$aiRequestBody = [
    'model' => 'gpt-4o-mini',
    'messages' => $aiMessages,
    'temperature' => 0.3,
    'max_tokens' => 10
];

$ch = curl_init($aiUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $OPENAI_API_KEY
    ],
    CURLOPT_POSTFIELDS => json_encode($aiRequestBody),
    CURLOPT_TIMEOUT => 30
]);

$aiResponse = curl_exec($ch);
curl_close($ch);

$aiData = json_decode($aiResponse, true);
$bestMatchIndex = 0;

if (isset($aiData['choices'][0]['message']['content'])) {
    $bestMatchIndex = (int) trim($aiData['choices'][0]['message']['content']) - 1;
}

if ($bestMatchIndex < 0 || $bestMatchIndex >= count($linkedInProfiles)) {
    http_response_code(404);
    echo json_encode(['error' => 'No suitable LinkedIn profile found', 'ai_response' => $aiData]);
    exit();
}

$bestProfile = $linkedInProfiles[$bestMatchIndex];

// Step 3: Use RocketReach
$linkedInUrl = $bestProfile['link'];
$rocketReachUrl = "https://api.rocketreach.co/v2/api/lookupProfile?li_url=" . urlencode($linkedInUrl);

$ch = curl_init($rocketReachUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => [
        'Api-Key: ' . $ROCKETREACH_API_KEY
    ],
    CURLOPT_TIMEOUT => 30
]);

$rocketReachResponse = curl_exec($ch);
$rocketReachHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

$contactInfo = [
    'linkedin_url' => $linkedInUrl,
    'profile_title' => $bestProfile['title'],
    'emails' => [],
    'phones' => [],
    'current_employer' => '',
    'location' => ''
];

$hasEmail = false;
$hasPhone = false;

if ($rocketReachHttpCode === 200 && $rocketReachResponse) {
    $rocketReachData = json_decode($rocketReachResponse, true);
    
    if (isset($rocketReachData['emails'])) {
        $contactInfo['emails'] = $rocketReachData['emails'];
        $hasEmail = !empty($contactInfo['emails']);
    }
    
    if (isset($rocketReachData['phones'])) {
        $contactInfo['phones'] = $rocketReachData['phones'];
        $hasPhone = !empty($contactInfo['phones']);
    }
    
    if (isset($rocketReachData['current_employer'])) {
        $contactInfo['current_employer'] = $rocketReachData['current_employer'];
    }
    
    if (isset($rocketReachData['location'])) {
        $contactInfo['location'] = $rocketReachData['location'];
    }
}

$finalResponse = [
    'success' => true,
    'supervisor_name' => $supervisorName,
    'linkedin_found' => true,
    'contact_info' => $contactInfo,
    'google_results_count' => count($linkedInProfiles),
    'ai_selected_index' => $bestMatchIndex + 1,
    'cached' => false
];

// Store in cache
if ($conn) {
    try {
        $stmt = $conn->prepare("
            INSERT INTO cache_linkedin_contacts 
            (supervisor_name, normalized_name, linkedin_url, contact_data, google_results_count, ai_selected_index, has_email, has_phone, expires_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            linkedin_url = VALUES(linkedin_url),
            contact_data = VALUES(contact_data),
            google_results_count = VALUES(google_results_count),
            ai_selected_index = VALUES(ai_selected_index),
            has_email = VALUES(has_email),
            has_phone = VALUES(has_phone),
            expires_at = VALUES(expires_at),
            created_at = CURRENT_TIMESTAMP
        ");
        
        $contactJson = json_encode($contactInfo);
        $expiresAt = getCacheExpiration();
        
        $stmt->execute([
            $supervisorName,
            $normalizedName,
            $linkedInUrl,
            $contactJson,
            count($linkedInProfiles),
            $bestMatchIndex + 1,
            $hasEmail ? 1 : 0,
            $hasPhone ? 1 : 0,
            $expiresAt
        ]);
    } catch (PDOException $e) {
        error_log("Failed to cache LinkedIn contact: " . $e->getMessage());
    }
}

http_response_code(200);
echo json_encode($finalResponse);
?>