<?php
require_once('db.php');

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!isset($data['slug']) || empty(trim($data['slug']))) {
    http_response_code(400);
    echo json_encode(['error' => 'slug is required']);
    exit();
}

$slug = trim($data['slug']);
$conn = getDbConnection();

// Check cache first
if ($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT profile_data, server_used, created_at 
            FROM cache_supervisor_profiles 
            WHERE supervisor_slug = ? 
            AND expires_at > NOW()
            AND success = 1
            LIMIT 1
        ");
        $stmt->execute([$slug]);
        $cached = $stmt->fetch();
        
        if ($cached) {
            updateCacheStats('fetch_supervisor_profile', true);
            $response = [
                'success' => true,
                'server_used' => $cached['server_used'],
                'profile' => json_decode($cached['profile_data'], true),
                'cached' => true,
                'cached_at' => $cached['created_at']
            ];
            http_response_code(200);
            echo json_encode($response);
            exit();
        }
        
        updateCacheStats('fetch_supervisor_profile', false);
    } catch (PDOException $e) {
        error_log("Cache check failed: " . $e->getMessage());
    }
}

// Cache miss - fetch from API
$servers = [
    'https://search.playlistsupply.com/tunefind/supervisor/',
    'https://health.playlistsupply.com/tunefind/supervisor/'
];
$randomServer = $servers[array_rand($servers)];
$url = $randomServer . $slug;

$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_FOLLOWLOCATION => true
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpCode !== 200 || !$response) {
    http_response_code(500);
    echo json_encode(['error' => 'Failed to fetch supervisor profile', 'server_used' => $randomServer]);
    exit();
}

$profileData = json_decode($response, true);

if (!isset($profileData['success']) || !$profileData['success']) {
    http_response_code(404);
    echo json_encode(['error' => 'Supervisor profile not found']);
    exit();
}

// Extract relevant info
$supervisorInfo = [
    'user' => [],
    'socials' => []
];

if (isset($profileData['data']['user'])) {
    $user = $profileData['data']['user'];
    $supervisorInfo['user'] = [
        'username' => $user['username'] ?? '',
        'name' => $user['name'] ?? '',
        'userSince' => $user['userSince'] ?? '',
        'profileImage' => $user['profileImage'] ?? '',
        'customProfileImage' => $user['customProfileImage'] ?? '',
        'isSuper' => $user['isSuper'] ?? false
    ];
}

if (isset($profileData['data']['user-socials']) && is_array($profileData['data']['user-socials'])) {
    foreach ($profileData['data']['user-socials'] as $social) {
        $supervisorInfo['socials'][] = [
            'title' => $social['title'] ?? '',
            'link' => $social['link'] ?? '',
            'icon' => $social['icon'] ?? ''
        ];
    }
}

$finalResponse = [
    'success' => true,
    'server_used' => $randomServer,
    'profile' => $supervisorInfo,
    'cached' => false
];

// Store in cache
if ($conn) {
    try {
        $stmt = $conn->prepare("
            INSERT INTO cache_supervisor_profiles 
            (supervisor_slug, profile_data, server_used, expires_at) 
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            profile_data = VALUES(profile_data),
            server_used = VALUES(server_used),
            expires_at = VALUES(expires_at),
            created_at = CURRENT_TIMESTAMP
        ");
        
        $profileJson = json_encode($supervisorInfo);
        $expiresAt = getCacheExpiration();
        
        $stmt->execute([
            $slug,
            $profileJson,
            $randomServer,
            $expiresAt
        ]);
    } catch (PDOException $e) {
        error_log("Failed to cache supervisor profile: " . $e->getMessage());
    }
}

http_response_code(200);
echo json_encode($finalResponse);
?>