<?php
require_once('db.php');

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!isset($data['appearance_link']) || empty(trim($data['appearance_link']))) {
    http_response_code(400);
    echo json_encode(['error' => 'appearance_link is required']);
    exit();
}

$appearanceLink = trim($data['appearance_link']);
// Remove leading slash if present
$appearanceLink = ltrim($appearanceLink, '/');

$conn = getDbConnection();

// Check cache first
if ($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT supervisors_data, server_used, supervisor_count, created_at 
            FROM cache_show_supervisors 
            WHERE appearance_link = ? 
            AND expires_at > NOW()
            AND success = 1
            LIMIT 1
        ");
        $stmt->execute([$appearanceLink]);
        $cached = $stmt->fetch();
        
        if ($cached) {
            updateCacheStats('fetch_show_supervisors', true);
            $supervisors = json_decode($cached['supervisors_data'], true);
            
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'server_used' => $cached['server_used'],
                'supervisors' => $supervisors,
                'count' => $cached['supervisor_count'],
                'cached' => true,
                'cached_at' => $cached['created_at']
            ]);
            exit();
        }
        
        updateCacheStats('fetch_show_supervisors', false);
    } catch (PDOException $e) {
        error_log("Cache check failed: " . $e->getMessage());
    }
}

// Cache miss - fetch from TuneFind
// Random server selection
$servers = [
    'https://search.playlistsupply.com/tunefind/',
    'https://health.playlistsupply.com/tunefind/'
];
$randomServer = $servers[array_rand($servers)];
$url = $randomServer . $appearanceLink;

// Fetch show/movie data
$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_FOLLOWLOCATION => true
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpCode !== 200 || !$response) {
    // Try the other server if first one fails
    $otherServer = ($randomServer === $servers[0]) ? $servers[1] : $servers[0];
    $url = $otherServer . $appearanceLink;
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200 || !$response) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to fetch show/movie data', 'tried_servers' => $servers]);
        exit();
    }
    
    $randomServer = $otherServer;
}

$showData = json_decode($response, true);

// Check for both possible data structures (data or apiData)
$dataKey = null;
if (isset($showData['data'])) {
    $dataKey = 'data';
} elseif (isset($showData['apiData'])) {
    $dataKey = 'apiData';
}

if (!$dataKey) {
    http_response_code(404);
    echo json_encode(['error' => 'Show/Movie data structure not recognized']);
    exit();
}

// Extract supervisors
$supervisors = [];
$mainData = $showData[$dataKey];

// Check if it's a show with seasons/episodes or a movie
if (isset($mainData['parents']) && is_array($mainData['parents'])) {
    // It's a show with seasons
    foreach ($mainData['parents'] as $season) {
        if (isset($season['supervisors']) && is_array($season['supervisors'])) {
            foreach ($season['supervisors'] as $supervisor) {
                $supervisors[] = [
                    'id' => $supervisor['id'] ?? '',
                    'name' => $supervisor['name'] ?? '',
                    'slug' => $supervisor['slug'] ?? '',
                    'achievements' => $supervisor['achievements'] ?? 0,
                    'season' => $season['name'] ?? 'Unknown Season'
                ];
            }
        }
    }
}

// Also check for supervisors at the main level (for movies or overall show supervisors)
if (isset($mainData['supervisors']) && is_array($mainData['supervisors'])) {
    foreach ($mainData['supervisors'] as $supervisor) {
        $supervisors[] = [
            'id' => $supervisor['id'] ?? '',
            'name' => $supervisor['name'] ?? '',
            'slug' => $supervisor['slug'] ?? '',
            'achievements' => $supervisor['achievements'] ?? 0,
            'season' => 'Overall'
        ];
    }
}

// Deduplicate supervisors by ID while keeping track of which seasons they worked on
$supervisorMap = [];
foreach ($supervisors as $supervisor) {
    if (!empty($supervisor['id'])) {
        if (!isset($supervisorMap[$supervisor['id']])) {
            $supervisorMap[$supervisor['id']] = $supervisor;
            $supervisorMap[$supervisor['id']]['seasons'] = [$supervisor['season']];
            unset($supervisorMap[$supervisor['id']]['season']);
        } else {
            // Add season to existing supervisor
            if (!in_array($supervisor['season'], $supervisorMap[$supervisor['id']]['seasons'])) {
                $supervisorMap[$supervisor['id']]['seasons'][] = $supervisor['season'];
            }
        }
    }
}

$uniqueSupervisors = array_values($supervisorMap);

// Store in cache
if ($conn && !empty($uniqueSupervisors)) {
    try {
        $stmt = $conn->prepare("
            INSERT INTO cache_show_supervisors 
            (appearance_link, supervisors_data, server_used, supervisor_count, success, expires_at) 
            VALUES (?, ?, ?, ?, 1, ?)
            ON DUPLICATE KEY UPDATE 
            supervisors_data = VALUES(supervisors_data),
            server_used = VALUES(server_used),
            supervisor_count = VALUES(supervisor_count),
            success = 1,
            expires_at = VALUES(expires_at),
            created_at = CURRENT_TIMESTAMP
        ");
        
        $supervisorsJson = json_encode($uniqueSupervisors);
        $expiresAt = getCacheExpiration();
        
        $stmt->execute([
            $appearanceLink,
            $supervisorsJson,
            $randomServer,
            count($uniqueSupervisors),
            $expiresAt
        ]);
    } catch (PDOException $e) {
        error_log("Failed to cache supervisors: " . $e->getMessage());
    }
}

http_response_code(200);
echo json_encode([
    'success' => true,
    'server_used' => $randomServer,
    'supervisors' => $uniqueSupervisors,
    'count' => count($uniqueSupervisors),
    'cached' => false
]);
?>