<?php
require_once('db.php');

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!isset($data['artist_name']) || empty(trim($data['artist_name']))) {
    http_response_code(400);
    echo json_encode(['error' => 'artist_name is required']);
    exit();
}

$artistName = trim($data['artist_name']);
$conn = getDbConnection();

// Check cache first
if ($conn) {
    try {
        $stmt = $conn->prepare("
            SELECT response_data, server_used, created_at 
            FROM cache_artist_shows 
            WHERE formatted_artist_name = ? 
            AND expires_at > NOW()
            AND success = 1
            LIMIT 1
        ");
        $stmt->execute([$artistName]);
        $cached = $stmt->fetch();
        
        if ($cached) {
            updateCacheStats('fetch_artist_shows', true);
            $response = json_decode($cached['response_data'], true);
            $response['cached'] = true;
            $response['cached_at'] = $cached['created_at'];
            http_response_code(200);
            echo json_encode($response);
            exit();
        }
        
        updateCacheStats('fetch_artist_shows', false);
    } catch (PDOException $e) {
        error_log("Cache check failed: " . $e->getMessage());
    }
}

// Cache miss - fetch from API
$servers = [
    'https://search.playlistsupply.com/tunefind/artist/',
    'https://health.playlistsupply.com/tunefind/artist/'
];
$randomServer = $servers[array_rand($servers)];
$url = $randomServer . urlencode($artistName);

$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_FOLLOWLOCATION => true
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpCode !== 200 || !$response) {
    http_response_code(500);
    echo json_encode(['error' => 'Failed to fetch artist data', 'server_used' => $randomServer]);
    exit();
}

$artistData = json_decode($response, true);

if (!isset($artistData['success']) || !$artistData['success']) {
    http_response_code(404);
    echo json_encode(['error' => 'Artist not found']);
    exit();
}

// Process the data
$processedData = [
    'artist_name' => $artistData['data']['name'] ?? $artistName,
    'artist_id' => $artistData['data']['id'] ?? null,
    'songs' => []
];

if (isset($artistData['data']['credits']) && is_array($artistData['data']['credits'])) {
    foreach ($artistData['data']['credits'] as $credit) {
        $song = [
            'name' => $credit['name'] ?? '',
            'slug' => $credit['slug'] ?? '',
            'id' => $credit['id'] ?? 0,
            'shows' => []
        ];
        
        if (isset($credit['appearances']) && is_array($credit['appearances'])) {
            foreach ($credit['appearances'] as $appearance) {
                $show = [
                    'title' => $appearance['title'] ?? '',
                    'description' => $appearance['description'] ?? '',
                    'airDate' => $appearance['airDate'] ?? '',
                    'appearanceLink' => $appearance['appearanceLink'] ?? '',
                    'scoreCalculated' => $appearance['scoreCalculated'] ?? 0,
                    'openForVote' => $appearance['openForVote'] ?? false,
                    'ir' => $appearance['ir'] ?? false,
                    'iw' => $appearance['iw'] ?? false
                ];
                $song['shows'][] = $show;
            }
        }
        
        $processedData['songs'][] = $song;
    }
}

$finalResponse = [
    'success' => true,
    'server_used' => $randomServer,
    'data' => $processedData,
    'cached' => false
];

// Store in cache
if ($conn) {
    try {
        $stmt = $conn->prepare("
            INSERT INTO cache_artist_shows 
            (artist_name, formatted_artist_name, response_data, server_used, expires_at) 
            VALUES (?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            response_data = VALUES(response_data),
            server_used = VALUES(server_used),
            expires_at = VALUES(expires_at),
            created_at = CURRENT_TIMESTAMP
        ");
        
        $originalName = $processedData['artist_name'];
        $responseJson = json_encode($finalResponse);
        $expiresAt = getCacheExpiration();
        
        $stmt->execute([
            $originalName,
            $artistName,
            $responseJson,
            $randomServer,
            $expiresAt
        ]);
    } catch (PDOException $e) {
        error_log("Failed to cache artist shows: " . $e->getMessage());
    }
}

http_response_code(200);
echo json_encode($finalResponse);
?>